<?php
/*
Plugin Name:     Post Expiration Unposter
Description:     Automatically unpublishes posts on their expiration date.
Version:         0.3
Author:          AlecWilcoxMedia
Text Domain:     post-expiration-unposter
License:         GPLv2 or later
License URI:     https://www.gnu.org/licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Activation: schedule daily event
 */
function peu_activate() {
    if ( ! wp_next_scheduled( 'peu_daily_event' ) ) {
        wp_schedule_event( time(), 'daily', 'peu_daily_event' );
    }
}
register_activation_hook( __FILE__, 'peu_activate' );

/**
 * Deactivation: clear scheduled event
 */
function peu_deactivate() {
    wp_clear_scheduled_hook( 'peu_daily_event' );
}
register_deactivation_hook( __FILE__, 'peu_deactivate' );


/**
 * Add expiration date meta box
 */
add_action( 'add_meta_boxes', 'peu_add_meta_box' );
function peu_add_meta_box() {
    add_meta_box(
        'peu_meta_box',
        esc_html__( 'Expiration Date', 'post-expiration-unposter' ),
        'peu_meta_box_callback',
        'post',
        'side',
        'default'
    );
}

function peu_meta_box_callback( $post ) {
    wp_nonce_field( 'peu_save_meta', 'peu_meta_nonce' );

    $value = get_post_meta( $post->ID, '_peu_expiration_date', true );

    echo '<label for="peu_expiration_date">'
         . esc_html__( 'Select expiration date:', 'post-expiration-unposter' )
         . '</label>';

    printf(
        '<input type="date" id="peu_expiration_date" name="peu_expiration_date" value="%s" style="width:100%%"/>',
        esc_attr( $value )
    );
}


/**
 * Save expiration date meta
 */
add_action( 'save_post', 'peu_save_meta_box_data' );
function peu_save_meta_box_data( $post_id ) {
    // 1) Bail on autosave
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
        return;
    }

    // 2) Capability check
    if ( ! current_user_can( 'edit_post', $post_id ) ) {
        return;
    }

    // 3) Unsplash + sanitize nonce in one go
    $nonce = isset( $_POST['peu_meta_nonce'] )
        ? sanitize_text_field( wp_unslash( $_POST['peu_meta_nonce'] ) )
        : '';

    // 4) Verify nonce
    if ( ! wp_verify_nonce( $nonce, 'peu_save_meta' ) ) {
        return;
    }

    // 5) Unsplash + sanitize expiration date in one go
    if ( isset( $_POST['peu_expiration_date'] ) ) {
        $date = sanitize_text_field( wp_unslash( $_POST['peu_expiration_date'] ) );

        if ( $date ) {
            update_post_meta( $post_id, '_peu_expiration_date', $date );
        } else {
            delete_post_meta( $post_id, '_peu_expiration_date' );
        }
    }
}


/**
 * Daily event hook: check and unpublish expired posts.
 */
add_action( 'peu_daily_event', 'peu_check_expired_posts' );
function peu_check_expired_posts() {
    global $wpdb;

    // Get today in WP timezone
    $today = date_i18n( 'Y-m-d', current_time( 'timestamp' ) );

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
    $post_ids = $wpdb->get_col(
        $wpdb->prepare(
            "
            SELECT p.ID
            FROM {$wpdb->posts}    AS p
            INNER JOIN {$wpdb->postmeta} AS pm
                ON p.ID = pm.post_id
            WHERE pm.meta_key   = %s
              AND pm.meta_value <= %s
              AND p.post_status  = 'publish'
              AND p.post_type    = 'post'
            ",
            '_peu_expiration_date',
            $today
        )
    );

    if ( ! empty( $post_ids ) ) {
        foreach ( $post_ids as $post_id ) {
            wp_update_post( array(
                'ID'          => $post_id,
                'post_status' => 'draft',
            ) );
        }
    }
}